/************************************************************************
ProcessHelper - A class to manipulate processes on Win32
Copyright (C) 2006 Daniel "Lesco" Schoepe

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
************************************************************************/

#include "ProcessHelper.h"

#include <vector>

using namespace std;

ProcessHelper::ProcessHelper()
{
	processOpened = false;
	this->process = 0;
}

ProcessHelper::~ProcessHelper()
{
	//kill threads which are still waiting for a bp to occur
	vector<THREAD_DATA>::iterator it;
	for( it = runningThreads.begin(); it != runningThreads.end(); ++it ) {
		TerminateThread(it->handle,1);
		delete it->data;
	}
}

ProcessHelper::ProcessHelper(HANDLE process)
{
	setProcess(process);
}


void ProcessHelper::setProcess(HANDLE process)
{
	processOpened = true;
	this->process = process;
}

HANDLE ProcessHelper::getProcess()
{
	if( !processOpened )
		throw (int)(NO_PROCESS_OPENED);
	return process;
}

DWORD ProcessHelper::readDword(DWORD addr)
{
	DWORD result;
	readData(addr,&result,sizeof(DWORD));
	return result;
}

WORD ProcessHelper::readWord(DWORD addr)
{
	WORD result;
	readData(addr,&result,sizeof(WORD));
	return result;
}

BYTE ProcessHelper::readByte(DWORD addr)
{
	BYTE result;
	readData(addr,&result,sizeof(BYTE));
	return result;
}

void ProcessHelper::readData(DWORD addr,void *buffer,DWORD length)
{
	const DWORD PAGE_SIZE = 0x1000; 
	if( !processOpened )
		throw (int)(NO_PROCESS_OPENED);
	DWORD result,bytesRead,oldProt,i,temp;
	temp = 0;
	bytesRead = 0;
	if( length > PAGE_SIZE ) {

		for( i = 0; i < (length/PAGE_SIZE); ++i ) {
			VirtualProtectEx(process,(void*)(addr+i*PAGE_SIZE),PAGE_SIZE,
				PAGE_READONLY,&oldProt);
			result = ReadProcessMemory(process,(void*)(addr+i*PAGE_SIZE),
				(char*)buffer+i*PAGE_SIZE,PAGE_SIZE,&temp);
			VirtualProtectEx(process,(void*)(addr+i*PAGE_SIZE),PAGE_SIZE,
				oldProt,&oldProt);
			bytesRead += temp;
		}
		
	}
	VirtualProtectEx(process,(void*)addr,length % PAGE_SIZE,
		PAGE_READONLY,&oldProt);
	result = ReadProcessMemory(process,(void*)addr,buffer,
		length % PAGE_SIZE,&temp);
	VirtualProtectEx(process,(void*)addr,length % PAGE_SIZE,
		oldProt,&oldProt);
	bytesRead += temp;
	if( !result || (bytesRead != length) ) {
		throw (int)(GetLastError());
	}
	return;	
}

void ProcessHelper::writeDword(DWORD addr,DWORD data)
{
	writeData(addr,&data,sizeof(DWORD));
}

void ProcessHelper::writeWord(DWORD addr,WORD data)
{
	writeData(addr,&data,sizeof(WORD));
}

void ProcessHelper::writeByte(DWORD addr,BYTE data)
{
	writeData(addr,&data,sizeof(BYTE));
}

void ProcessHelper::writeData(DWORD addr,void *buffer,DWORD length)
{
	if( !processOpened )
		throw (int)(NO_PROCESS_OPENED);
	DWORD result,bytesWritten,oldProt;
	VirtualProtectEx(process,(void*)addr,length,PAGE_READWRITE,&oldProt);
	result = WriteProcessMemory(process,(void*)addr,buffer,length,
			&bytesWritten);
	VirtualProtectEx(process,(void*)addr,length,oldProt,&oldProt);
	if( !result || (bytesWritten != length) ) {
		throw (int)(GetLastError());
	}
	return;	
}

void ProcessHelper::addBreakpoint(DWORD addr)
{
	if( !processOpened )
		throw (int)(NO_PROCESS_OPENED);
	const WORD LOOP=0xFEEB;
	vector<BREAKPOINT>::iterator it;
	for( it = breakpoints.begin(); it != breakpoints.end(); it++ ) {
		if( (it->addr == addr) || (it->addr == (addr+1)) ||
			(it->addr == (addr-1)) ) {
			throw (int)(BP_ALREADY_SET);
		}
	}
	BREAKPOINT bp;
	bp.addr = addr;
	bp.oldBytes = readWord(addr);
	breakpoints.push_back(bp);
	writeWord(addr,LOOP);
}

void ProcessHelper::addBreakpointAndRun(DWORD addr,BP_HANDLER handler,
					HANDLE thread)
{
	addBreakpoint(addr);
	BP_DATA *bpdata;
	THREAD_DATA tdata;
	bpdata = new BP_DATA;
	bpdata->addr = addr;
	bpdata->thread = thread;
	bpdata->ph = this;
	bpdata->handler = handler;
	bpdata->bpThreads = &runningThreads;
	tdata.data = bpdata;
	tdata.handle = CreateThread(0,0,(LPTHREAD_START_ROUTINE)waitForBpProc,
				bpdata,	0,&tdata.tid);
	if( !(tdata.handle) ) {
		delete bpdata;
		throw (int)(GetLastError());	
	}
	runningThreads.push_back(tdata);
	ResumeThread(thread);
}

void ProcessHelper::removeBreakpoint(DWORD addr)
{
	if( !processOpened )
		throw (int)(NO_PROCESS_OPENED);
	vector<BREAKPOINT>::iterator it;

	for( it = breakpoints.begin(); it != breakpoints.end(); it++ ) {
		if( it->addr == addr ) {
			writeWord(it->addr,it->oldBytes);
			breakpoints.erase(it);
			break;
		}
	}
}

void ProcessHelper::waitForBreakpoint(DWORD addr,HANDLE thread,CONTEXT& ctx)
{
	if( !processOpened )
		throw (int)(NO_PROCESS_OPENED);
	ctx.ContextFlags = CONTEXT_INTEGER | CONTEXT_CONTROL;
	ResumeThread(thread);
	do {
		GetThreadContext(thread,&ctx);
		Sleep(1);
	}
	while( ctx.Eip != addr );
	SuspendThread(thread);
}

DWORD __stdcall ProcessHelper::waitForBpProc(BP_DATA *bpdata)
{
	CONTEXT ctx;
	vector<THREAD_DATA>::iterator it;
	vector<THREAD_DATA> *threads;
	DWORD thisThreadId;
	thisThreadId = GetCurrentThreadId();
	ctx.ContextFlags = CONTEXT_INTEGER | CONTEXT_CONTROL;
	do {
		Sleep(10);
		GetThreadContext(bpdata->thread,&ctx);
	}
	while( ctx.Eip != bpdata->addr ); 
	SuspendThread(bpdata->thread);
	(*(bpdata->handler))(bpdata->ph,ctx);
	threads = bpdata->bpThreads;
	for( it = threads->begin(); it != threads->end(); ++it ) {
		if( it->tid == thisThreadId ) {
			threads->erase(it);
			break;
		}
	}
	delete bpdata;
	return 0;
}

void ProcessHelper::injectLibrary(char *libname)
{
	if( !processOpened )
		throw (int)(NO_PROCESS_OPENED);
	char *mem;
	DWORD tid;
	int length;
	const int CODE_SIZE = 13;
	DWORD loadLib = (DWORD)GetProcAddress(GetModuleHandle("kernel32"),
					"LoadLibraryA");
	BYTE code[] = { 0x68,0,0,0,0,	//push x
			0xE8,0,0,0,0,	//call x
			0xC2,4,0 }; //retn 4
	length = strlen(libname);
	mem = (char*)VirtualAllocEx(process,0,CODE_SIZE+length+1,
				MEM_COMMIT,PAGE_EXECUTE_READWRITE);
	if( !mem )
		throw (int)(GetLastError());
	*((DWORD*)(code + 1)) = (DWORD)mem;
	*((DWORD*)(code + 6)) = loadLib - ((DWORD)mem + length + 1) - 10;
	writeData((DWORD)mem,libname,length+1);
	writeData((DWORD)mem+length+1,code,CODE_SIZE);
	if( !CreateRemoteThread(process,0,0,
		(LPTHREAD_START_ROUTINE)(mem+length+1),0,0,&tid) ) {
		throw (int)(GetLastError());
	}
}

DWORD ProcessHelper::getImageSize(DWORD imageBase)
{
	if( !processOpened )
		throw (int)(NO_PROCESS_OPENED);
	DWORD e_lfanew;
	IMAGE_NT_HEADERS ntHeaders;
	e_lfanew = readDword(imageBase+60);
	readData(imageBase+e_lfanew,&ntHeaders,sizeof(IMAGE_NT_HEADERS));
	return ntHeaders.OptionalHeader.SizeOfImage;
}

DWORD ProcessHelper::searchPattern(DWORD start,const char *pattern,DWORD size)
{
	//validate pattern:
	char c;
	DWORD patternLength = strlen(pattern);
	int i,j;
	DWORD imageBase;
	if( patternLength & 1 )
		throw (int)(INVALID_PARAMETER);

	for( i = 0; i < patternLength; ++i ) {
		c = toupper(pattern[i]);
		if( ((c < 'A') || (c > 'F')) && 
			((c < '0') || (c > '9')) && (c != '?') ) {
			throw (int)(INVALID_PARAMETER);
		}
	}
	//find imageBase
	imageBase = start & 0xFFFF0000;
	while( (readWord(imageBase) != IMAGE_DOS_SIGNATURE) ) {
		imageBase -= 0x10000;
	}
	if( !size ) {
		
		size = getImageSize(imageBase);
	}
	//copy whole area into our process memory:
	char *buffer = new char[size];
	readData(start,buffer,size);
	BYTE *patternBytes = new BYTE[patternLength/2];
	char temp[3];
	temp[2] = 0;
	for( i = 0,j = 0; i < patternLength; i+=2,++j ) {
		memcpy(temp,pattern+i,2);
		if( temp[0] == '?' )
			temp[0] = '0';
		if( temp[1] == '?' )
			temp[1] = '0';
		patternBytes[j] = strtoul(temp,0,16);
	}
	BYTE current,inprocess;
	DWORD limit = start + size - (patternLength/2);
	//TODO: use a more efficient algorithm
	for( i = start; i < limit;  ) {
		//find mismatch:
		for( j = 0; j < patternLength/2; ++j ) {
			current = patternBytes[j];
			inprocess = buffer[(i-start)+j];
			if( pattern[j*2] == '?' ) {
				inprocess &= 0xF;
				current &= 0xF;
			}
			if( pattern[j*2+1] == '?' ) {
				inprocess &= 0xF0;
				current &= 0xF0;
			}
			if( current != inprocess ) {
				++i;
				break;
			}
		}
		if( j != (patternLength/2) ) {
			continue;
		}
		else {
			delete [] patternBytes;
			delete [] buffer;
			return i;
		}
	}
	delete [] patternBytes;
	delete [] buffer;
	return 0;
}

void ProcessHelper::dumpProcess(DWORD imageBase,const char *filename)
{
	if( !processOpened )
		throw (int)(NO_PROCESS_OPENED);
	DWORD bytesProcessed,totalSize,result;
	totalSize = getImageSize(imageBase);
	HANDLE file = CreateFile(filename,GENERIC_WRITE,0,0,CREATE_ALWAYS,0,0);
	if( file == INVALID_HANDLE_VALUE ) {
		throw (int)(GetLastError());
	}
	char *buffer = new char[totalSize];
	readData(imageBase,buffer,totalSize);
	result = WriteFile(file,buffer,totalSize,&bytesProcessed,0);
	delete [] buffer;
	CloseHandle(file);
	if( !result || (bytesProcessed != totalSize) ) {
		throw (int)(GetLastError());
	}
	PEFile pe;
	pe.openFile(filename);
	DWORD sectCount = pe.ntHeaders->FileHeader.NumberOfSections,i;
	IMAGE_SECTION_HEADER* section;
	for( i = 0; i < sectCount; i++ ) {
		section = pe.sectionHeader(i);
		section->PointerToRawData = section->VirtualAddress;
		section->SizeOfRawData = section->Misc.VirtualSize;
	}	
	pe.close();
	
}


char * ProcessHelper::formatErrorMessage(int errorcode)
{
	char *result;
	if( errorcode > 0 ) {
		//error from win32-api:
		FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
				FORMAT_MESSAGE_FROM_SYSTEM,
				0,errorcode,0,(char*)&result,0,0);
	}
	else {
		char *msg;
		switch( errorcode ) {
		case BP_ALREADY_SET:
			msg = "Breakpoint at this address already set";
			break;
		case NO_PROCESS_OPENED:
			msg = "No process opened";
			break;
		case INVALID_PARAMETER:
			msg = "Invalid parameter";
			break;
		default:
			msg = "An error occured(I know this message "
				"doesn't help you)";
			
		}
		result = (char*)LocalAlloc(LMEM_FIXED|LMEM_ZEROINIT,
					strlen(msg)+1);
		strcpy(result,msg);
	}
	return result;
}

